<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Mail\CareersMailManager;
use App\Mail\ProposeMailManager;
use Illuminate\Http\Request;
use Mail;
use App\Mail\ContactMailManager;
use Cache;
use App\Models\Category;
use App\Rules\Recaptcha;
use Illuminate\Validation\Rule;

class ContactController extends Controller
{

    public function __construct()
    {

    }

    public function index()
    {
        $featured_categories = Cache::rememberForever('featured_categories', function () {
            return Category::with('bannerImage')->where('featured', 1)->orderBy('order_level', 'asc')->get();
        });

        $all_categories = Cache::rememberForever('all_categories', function () {
            return Category::with('bannerImage')->orderBy('order_level', 'asc')->get();
        });

        return view('frontend.contact.index', compact('featured_categories', 'all_categories'));
    }

    public function process(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required',
            'phone' => 'required',
            'subject' => 'required',
            'message' => 'required',
        ]);

        $subjects = [
            1 => translate('I don\'t receive today\'s emails'),
            2 => translate('Problems during purchase'),
            3 => translate('I forgot my password'),
            4 => translate('Complaints'),
            5 => translate('Suggestions'),
            6 => translate('Invoices'),
            7 => translate('Others'),
        ];

        $validated['subject'] = $subjects[$validated['subject']];

        $validated['template'] = 'emails.contact_admin';
        Mail::to('contact@deelgo.com')->queue(new ContactMailManager($validated));

        $validated['template'] = 'emails.contact_customer';
        $validated['subject'] = 'Deelgo | ' . translate('Confirmation of receipt of your request from the contact form');
        Mail::to($validated['email'])->queue(new ContactMailManager($validated));

        flash(translate('Your email has been sent'))->success();
        return back();
    }

    public function process_contact_form(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required',
            'phone' => 'required',
            'subject' => 'required',
            'message' => 'required',
            'g-recaptcha-response' => [
                Rule::when(get_setting('google_recaptcha') == 1, ['required', new Recaptcha()], ['sometimes'])
            ],
        ]);

        $subjects = [
            1 => translate('I don\'t receive today\'s emails'),
            2 => translate('Problems during purchase'),
            3 => translate('I forgot my password'),
            4 => translate('Complaints'),
            5 => translate('Suggestions'),
            6 => translate('Invoices'),
            7 => translate('Others'),
        ];

        $validated['subject'] = $subjects[$validated['subject']];

        $validated['template'] = 'emails.contact_admin';
        Mail::to('contact@deelgo.com')->queue(new ContactMailManager($validated));

        $validated['template'] = 'emails.contact_customer';
        $validated['subject'] = 'Deelgo | ' . translate('Confirmation of receipt of your request from the contact form');
        Mail::to($validated['email'])->queue(new ContactMailManager($validated));

        flash(translate('Your email has been sent'))->success();
        return back();;
    }

    public function process_propose_deal_form(Request $request)
    {
        $validated = $request->validate([
            'deal' => 'required',
            'manager' => 'required',
            'email' => 'required|email',
            'phone' => 'required',
            'city' => 'required',
            'message' => 'required',
            'g-recaptcha-response' => [
                Rule::when(get_setting('google_recaptcha') == 1, ['required', new Recaptcha()], ['sometimes'])
            ],
        ]);

        $validated['subject'] = translate('Objet') .':'. translate('Nouveau deal soumis – Vérification requise');
        $validated['template'] = 'emails.propose';

        $validated['content'] = $validated['subject'];
        $validated['content'] .= '<p>' . translate('Bonjour,') . '</p>';
        $validated['content'] .= '<p>' . translate('Un nouveau deal a été soumis sur la plateforme et nécessite une vérification avant approbation.') . '</p>';
        $validated['content'] .= '<ul>';
        $validated['content'] .= '<li>' . translate('Titre du dea') . ': <strong>' . $validated['deal'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Ajouté par') . ': <strong>' . $validated['manager'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Email') . ': <strong>' . $validated['email'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Téléphone ') . ': <strong>' . $validated['phone'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Ville') . ': <strong>' . $validated['city'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Détails') . ': <br><strong>' . $validated['message'] . '</strong></li>';
        $validated['content'] .= '</ul>';
        $validated['content'] .= '<p>' . translate('Merci d’examiner ce deal et de procéder à son approbation ou à toute action nécessaire.') . '</p>';
        $validated['content'] .= '<p>' . translate('Ceci est un message automatique envoyé par la plateforme.') . '</p>';

        Mail::to('deals@deelgo.com')->queue(new ProposeMailManager($validated));

        $validated['subject'] = 'Deelgo | ' . translate('Confirmation of receipt of your request from the deal form');

        $validated['content'] = '<strong style="color: #5f212e;">' . translate('Subject') . ' :</strong> ' . $validated['subject'];
        $validated['content'] = '<p>' . translate('We confirm receipt of your request for proposition deal.') . '</p>';

        Mail::to($validated['email'])->queue(new ProposeMailManager($validated));

        flash(translate('Your email has been sent'))->success();
        return back();
    }

    public function process_propose_event_form(Request $request)
    {
        $validated = $request->validate([
            'event' => 'required',
            'organizer' => 'required',
            'email' => 'required|email',
            'phone' => 'required',
            'city' => 'required',
            'message' => 'required',
            'g-recaptcha-response' => [
                Rule::when(get_setting('google_recaptcha') == 1, ['required', new Recaptcha()], ['sometimes'])
            ],
        ]);

        $validated['subject'] =  translate('Objet') . ':' . translate('Nouvel événement soumis – Vérification requise');
        $validated['template'] = 'emails.propose';

        $validated['content'] =  $validated['subject'];
        $validated['content'] .=  translate('Bonjour,');
        $validated['content'] .= '<p>' . translate('Un nouvel événement a été soumis sur la plateforme et nécessite une vérification avant approbation.') .'</p>';
        $validated['content'] .= '<ul>';
        $validated['content'] .= '<li>' . translate('Nom de l’événement') . ': <strong>' . $validated['event'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Ajouté par') . ': <strong>' . $validated['organizer'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Email') . ': <strong>' . $validated['email'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Téléphone') . ': <strong>' . $validated['phone'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Lieu ') . ': <strong>' . $validated['city'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Détails ') . ': <br><strong>' . $validated['message'] . '</strong></li>';
        $validated['content'] .= '</ul>';
        $validated['content'] .= translate('Merci d’examiner cet événement et de procéder à son approbation ou à toute action nécessaire.');
        $validated['content'] .= translate('Ceci est un message automatique envoyé par la plateforme.
');

        Mail::to('event@deelgo.com')->queue(new ProposeMailManager($validated));

        $validated['subject'] = 'Deelgo | ' . translate('Confirmation of receipt of your request from the event form');

        $validated['content'] = '<strong style="color: #5f212e;">' . translate('Subject') . ' :</strong> ' . $validated['subject'];
        $validated['content'] = '<p>' . translate('We confirm receipt of your request for proposition event.') . '</p>';

        Mail::to($validated['email'])->queue(new ProposeMailManager($validated));

        flash(translate('Your email has been sent'))->success();
        return back();
    }

    public function process_careers_form(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required|email',
            'phone' => 'required',
            'file' => 'required|file',
            'message' => 'required|string',
            'g-recaptcha-response' => [
                Rule::when(get_setting('google_recaptcha') == 1, ['required', new Recaptcha()], ['sometimes'])
            ],
        ]);

        $filePath = null;
        // Handle file upload 
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $destinationPath = 'uploads/cv';
            $fileName = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path($destinationPath), $fileName);
            $filePath = public_path($destinationPath) . '/' . $fileName;
        }

        unset($validated['file']);

        $validated['subject'] = translate('Objet ') . ':' . translate('Nouvelle candidature reçue – Espace Recrutement');
        $validated['template'] = 'emails.careers';

        $validated['content'] = $validated['subject'];
        $validated['content'] .= '<p>'. translate('Bonjour,') .'</p>';
        $validated['content'] .= '<p>' . translate('Une nouvelle candidature a été soumise via la plateforme de recrutement.') . '</p>';
        $validated['content'] .= '<ul>';
        $validated['content'] .= '<li>' . translate('Prénom') . ': <strong>' . $validated['first_name'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Nom') . ': <strong>' . $validated['last_name'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Email') . ': <strong>' . $validated['email'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Téléphone ') . ': <strong>' . $validated['phone'] . '</strong></li>';
        $validated['content'] .= '<li>' . translate('Message') . ': <br><strong>' . $validated['message'] . '</strong></li>';
        $validated['content'] .= '</ul>';
        $validated['content'] .= translate('Veuillez examiner cette candidature et procéder aux vérifications nécessaires. Une confirmation est requise une fois l’évaluation terminée.');
        $validated['content'] .= translate('Merci.');

        Mail::to('recrut@deelgo.com')->queue(new CareersMailManager($validated, $filePath));

        $validated['subject'] = 'Deelgo | ' . translate('Confirmation of receipt of your submission.');

        $validated['content'] = '<strong style="color: #5f212e;">' . translate('Subject') . ' :</strong> ' . $validated['subject'];
        $validated['content'] = '<p>' . translate('We confirm receipt of your submission.') . '</p>';

        Mail::to($validated['email'])->queue(new CareersMailManager($validated));

        flash(translate('Your email has been sent'))->success();
        return back();
    }
}