<?php
namespace App\Http\Controllers;
use App\Models\City;
use Cache;
use Illuminate\Http\Request;
use App\Models\Search;
use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Color;
use App\Models\Shop;
use App\Models\Attribute;
use App\Models\AttributeCategory;
use App\Utility\CategoryUtility;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
class SearchController extends Controller
{
    public function index(Request $request, $category_id = null, $brand_id = null)
    {
        if ($request->type) {
            session(['type' => $request->type]);
        }

        $min_price = $request->min_price;
        $max_price = $request->max_price;
        $seller_id = $request->seller_id;
        $attributes = Attribute::all();
        $selected_attribute_values = array();
        $colors = Color::all();
        $selected_color = null;
        $category = [];
        $categories = [];
        $conditions = [];

        $promotion = $request->promotion;

        $products = Product::where($conditions);

        if ($category_id != null) {
            $category_ids = CategoryUtility::children_ids($category_id);
            $category_ids[] = $category_id;
            $category = Category::with('childrenCategories')->find($category_id);

            $products = $category->products();

            $attribute_ids = AttributeCategory::whereIn('category_id', $category_ids)->pluck('attribute_id')->toArray();
            $attributes = Attribute::whereIn('id', $attribute_ids)->get();
        } else {
            $categories = Category::with('childrenCategories', 'coverImage')->where('level', 0)->orderBy('order_level', 'desc')->get();
        }

        // $products->whereHas('stocks', function ($query) {
        //     $query->where('qty', '>', 0);
        // });

        $query = $request->keyword;
        if ($query != null) {
            $searchController = new SearchController;
            $searchController->store($request);
            $products->where(function ($q) use ($query) {
                foreach (explode(' ', trim($query)) as $word) {
                    $q->where('name', 'like', '%' . $word . '%')
                        ->orWhere('tags', 'like', '%' . $word . '%')
                        ->orWhere('unit_price', 'like', '%' . $word . '%')
                        ->orWhereHas('product_translations', function ($q) use ($word) {
                            $q->where('name', 'like', '%' . $word . '%');
                        })
                        ->orWhereHas('stocks', function ($q) use ($word) {
                            $q->where('sku', 'like', '%' . $word . '%');
                        });
                }
            });
            $case1 = $query . '%';
            $case2 = '%' . $query . '%';
            $products->orderByRaw("CASE
                WHEN name LIKE '$case1' THEN 1
                WHEN name LIKE '$case2' THEN 2
                ELSE 3
                END");
        }

        $location = $request->location;
        if ($location != null) {
            $products->where('event_city', 'like', '%' . $location . '%');
        }

        // Categories
        $categories = $request->category;
        if ($categories) {
            $products->whereHas('categories', function ($query) use ($categories) {
                $query->whereIn('id', $categories);
            });
        }

        // Prices 
        $prices = $request->price;

        // Cities
        $cities = $request->city;
        if ($cities) {
            $products->whereIn('location', $cities);
        }

        // Ratings 
        $ratings = $request->rating;
        if ($ratings) {
            // $products->whereIn('rating', $ratings);  
            $products->whereIn('default_reviews', $ratings);
        }

        // Promotions
        if ($promotion) {
            $products->whereNotNull('discount_start_date')
                // ->whereDate(DB::raw('FROM_UNIXTIME(discount_start_date)'), '>=', Carbon::today())    
                ->whereHas('stocks', function ($query) {
                    $query->where('qty', '>', 0);
                });
        }

        // Sort
        $sort_by = $request->sort_by;
        switch ($sort_by) {
            case 'newest':
                $products->orderBy('created_at', 'desc');
                break;
            case 'oldest':
                $products->orderBy('created_at', 'asc');
                break;
            case 'price-asc':
                $products->orderBy('unit_price', 'asc');
                break;
            case 'price-desc':
                $products->orderBy('unit_price', 'desc');
                break;
            default:
                $products->orderBy('created_at', 'desc');
                break;
        }

        $products_count = $products->where('nature', session('type'))->count();

        $products = filter_products($products)->with('taxes')->paginate(12)->appends(request()->query());

        $productsAttrs = $products->pluck('attributes')->flatMap(function ($attrsIds) {
            return json_decode($attrsIds, true);
        })->unique()->values()->toArray();

        $productsAttributes = Attribute::whereIn('id', $productsAttrs)->get();

        $productsOptions = $products->pluck('choice_options')->flatMap(function ($attrOptions) {
            $options = json_decode($attrOptions, true);
            $values = [];
            foreach ($options as $option) {
                $values = array_merge($values, $option['values']);
            }
            return $values;
        })->unique()->values()->toArray();

        $min_price = $products->min('unit_price');
        $max_price = $products->max('unit_price');

        $productIds = $products->pluck('id')->toArray();
        $productsCities = City::whereHas('products', function ($query) use ($productIds) {
            $query->whereIn('id', $productIds);
        })->get();

        $featured_categories = Cache::rememberForever('featured_categories', function () {
            return Category::with('bannerImage')->where('featured', 1)->orderBy('order_level', 'asc')->get();
        });

        $all_categories = Cache::rememberForever('all_categories', function () {
            return Category::with('bannerImage')->orderBy('order_level', 'asc')->get();
        });

        return view('frontend.shop.index', compact('featured_categories', 'all_categories', 'products', 'products_count', 'location', 'query', 'category', 'categories', 'category_id', 'brand_id', 'sort_by', 'seller_id', 'min_price', 'max_price', 'attributes', 'selected_attribute_values', 'colors', 'selected_color', 'productsCities', 'productsAttributes', 'productsOptions'));
        // return view('frontend.product_listing', compact('featured_categories', 'all_categories', 'products', 'location', 'query', 'category', 'categories', 'category_id', 'brand_id', 'sort_by', 'seller_id', 'min_price', 'max_price', 'attributes', 'selected_attribute_values', 'colors', 'selected_color', 'productsCities'));
    }

    public function searchByAjax(Request $request, $category_id = null)
    {
        $sort_by = $request->sort_by;
        $availability = $request->availability;
        $min_price = $request->min_price;
        $max_price = $request->max_price;
        $ratings = $request->ratings;
        $cities = $request->cities;
        $selectedAttributes = $request->selectedAttributes;

        $products = Product::where([]);

        if ($category_id != null) {
            $category_ids = CategoryUtility::children_ids($category_id);
            $category_ids[] = $category_id;
            $category = Category::with('childrenCategories')->find($category_id);

            $products = $category->products();

            $attribute_ids = AttributeCategory::whereIn('category_id', $category_ids)->pluck('attribute_id')->toArray();
            $attributes = Attribute::whereIn('id', $attribute_ids)->get();
        } else {
            $categories = Category::with('childrenCategories', 'coverImage')->where('level', 0)->orderBy('order_level', 'desc')->get();
        }

        // Availability 
        if ($availability) {
            if ($availability === 'available') {
                $products->whereHas('stocks', function ($query) {
                    $query->where('qty', '>', 0);
                })
                    ->whereDate(DB::raw('FROM_UNIXTIME(event_start_date)'), '>=', Carbon::today());
            } elseif ($availability === 'end') {
                $products->whereDate(DB::raw('FROM_UNIXTIME(event_start_date)'), '<', Carbon::today());
            } elseif ($availability === 'out-of-stock') {
                $products->whereHas('stocks', function ($query) {
                    $query->where('qty', '<=', 0);
                });
            }
        }

        // Price
        if ($min_price || $max_price) {
            $products->whereBetween('unit_price', [$min_price, $max_price]);
        }

        // Ratings
        if ($ratings) {
            $products->whereIn('rating', $ratings);
        }

        // Cities
        if ($cities) {
            $products->whereIn('location', $cities);
        }

        // Attributes
        if ($selectedAttributes) {
            foreach ($selectedAttributes as $attribute_id => $values) {
                foreach ($values as $value) {
                    $products->whereJsonContains('choice_options', [
                        'attribute_id' => (string) $attribute_id,
                        'values' => [$value]
                    ]);
                }
            }
        }


        // Sort 
        if ($sort_by) {
            switch ($sort_by) {
                case 'newest':
                    $products->orderBy('created_at', 'desc');
                    break;
                case 'oldest':
                    $products->orderBy('created_at', 'asc');
                    break;
                case 'price-asc':
                    $products->orderBy('unit_price', 'asc');
                    break;
                case 'price-desc':
                    $products->orderBy('unit_price', 'desc');
                    break;
                case 'alphabet-asc':
                    $products->orderBy('name', 'asc');
                    break;
                case 'alphabet-desc':
                    $products->orderBy('name', 'desc');
                    break;
                default:
                    $products->orderBy('id', 'desc');
                    break;
            }
        }

        $products_count = $products->where('nature', 'event')->count();

        $products = filter_products($products)->with('taxes')->paginate(12);

        $productHtml = [];
        foreach ($products as $product) {
            $productHtml[] = '<div class="col custompad-10 has-transition z-1">' . view('frontend.partials.boxes.product_box_1', ['product' => $product])->render() . '</div>';
        }

        return response()->json([
            'products' => $productHtml,
            'next_page_url' => $products->nextPageUrl(),
            'has_more_pages' => $products->hasMorePages(),
            'count' => $products_count,
        ]);
    }

    public function loadMore(Request $request, $category_id = null)
    {
        $page = $request->page;
        $sort_by = $request->sort_by;
        $availability = $request->availability;
        $min_price = $request->min_price;
        $max_price = $request->max_price;
        $ratings = $request->ratings;
        $cities = $request->cities;

        $products = Product::where([]);

        if ($category_id != null) {
            $category_ids = CategoryUtility::children_ids($category_id);
            $category_ids[] = $category_id;
            $category = Category::with('childrenCategories')->find($category_id);

            $products = $category->products();

            $attribute_ids = AttributeCategory::whereIn('category_id', $category_ids)->pluck('attribute_id')->toArray();
            $attributes = Attribute::whereIn('id', $attribute_ids)->get();
        } else {
            $categories = Category::with('childrenCategories', 'coverImage')->where('level', 0)->orderBy('order_level', 'desc')->get();
        }

        // Availability 
        if ($availability) {
            if ($availability === 'available') {
                $products->whereHas('stocks', function ($query) {
                    $query->where('qty', '>', 0);
                })
                    ->whereDate(DB::raw('FROM_UNIXTIME(event_start_date)'), '>=', Carbon::today());
            } elseif ($availability === 'end') {
                $products->whereDate(DB::raw('FROM_UNIXTIME(event_start_date)'), '<', Carbon::today());
            } elseif ($availability === 'out-of-stock') {
                $products->whereHas('stocks', function ($query) {
                    $query->where('qty', '<=', 0);
                });
            }
        }

        // Price
        if ($min_price || $max_price) {
            $products->whereBetween('unit_price', [$min_price, $max_price]);
        }

        // Ratings
        if ($ratings) {
            $products->whereIn('rating', $ratings);
        }

        // Cities
        if ($cities) {
            $products->whereIn('location', $cities);
        }

        // Sort 
        if ($sort_by) {
            switch ($sort_by) {
                case 'newest':
                    $products->orderBy('created_at', 'desc');
                    break;
                case 'oldest':
                    $products->orderBy('created_at', 'asc');
                    break;
                case 'price-asc':
                    $products->orderBy('unit_price', 'asc');
                    break;
                case 'price-desc':
                    $products->orderBy('unit_price', 'desc');
                    break;
                case 'alphabet-asc':
                    $products->orderBy('name', 'asc');
                    break;
                case 'alphabet-desc':
                    $products->orderBy('name', 'desc');
                    break;
                default:
                    $products->orderBy('id', 'desc');
                    break;
            }
        }

        $products = filter_products($products)->with('taxes')->paginate(12);

        $productHtml = [];
        foreach ($products as $product) {
            $productHtml[] = '<div class="col custompad-10 has-transition z-1">' . view('frontend.partials.boxes.product_box_1', ['product' => $product])->render() . '</div>';
        }

        return response()->json([
            'products' => $productHtml,
            'next_page_url' => $products->nextPageUrl(),
            'has_more_pages' => $products->hasMorePages(),
            'count' => $products->count(),
        ]);
    }

    public function loadMoreProducts(Request $request)
    {
        $page = $request->page;
        $category_id = $request->category_id;
        $min_price = $request->min_price;
        $max_price = $request->max_price;
        $seller_id = $request->seller_id;
        $attributes = Attribute::all();
        $selected_attribute_values = array();
        $colors = Color::all();
        $selected_color = null;
        $category = [];
        $categories = [];
        $conditions = [];

        $products = Product::where($conditions);

        if ($category_id != null) {
            $category_ids = CategoryUtility::children_ids($category_id);
            $category_ids[] = $category_id;
            $category = Category::with('childrenCategories')->find($category_id);

            $products = $category->products();

            $attribute_ids = AttributeCategory::whereIn('category_id', $category_ids)->pluck('attribute_id')->toArray();
            $attributes = Attribute::whereIn('id', $attribute_ids)->get();
        } else {
            $categories = Category::with('childrenCategories', 'coverImage')->where('level', 0)->orderBy('order_level', 'desc')->get();
        }

        $products->whereHas('stocks', function ($query) {
            $query->where('qty', '>', 0);
        });

        $query = $request->keyword;
        if ($query != null) {
            $searchController = new SearchController;
            $searchController->store($request);
            $products->where(function ($q) use ($query) {
                foreach (explode(' ', trim($query)) as $word) {
                    $q->where('name', 'like', '%' . $word . '%')
                        ->orWhere('tags', 'like', '%' . $word . '%')
                        ->orWhere('unit_price', 'like', '%' . $word . '%')
                        ->orWhereHas('product_translations', function ($q) use ($word) {
                            $q->where('name', 'like', '%' . $word . '%');
                        })
                        ->orWhereHas('stocks', function ($q) use ($word) {
                            $q->where('sku', 'like', '%' . $word . '%');
                        });
                }
            });
            $case1 = $query . '%';
            $case2 = '%' . $query . '%';
            $products->orderByRaw("CASE
                WHEN name LIKE '$case1' THEN 1
                WHEN name LIKE '$case2' THEN 2
                ELSE 3
                END");
        }

        $location = $request->location;
        if ($location != null) {
            $products->where('event_city', 'like', '%' . $location . '%');
        }

        // Categories
        $categories = $request->category;
        if ($categories) {
            $products->whereHas('categories', function ($query) use ($categories) {
                $query->whereIn('id', $categories);
            });
        }

        // Prices 
        $prices = $request->price;

        // Cities
        $cities = $request->city;
        if ($cities) {
            $products->whereIn('location', $cities);
        }

        // Ratings 
        $ratings = $request->rating;
        if ($ratings) {
            // $products->whereIn('rating', $ratings);  
            $products->whereIn('default_reviews', $ratings);
        }

        // Sort
        $sort_by = $request->sort_by;
        switch ($sort_by) {
            case 'newest':
                $products->orderBy('created_at', 'desc');
                break;
            case 'oldest':
                $products->orderBy('created_at', 'asc');
                break;
            case 'price-asc':
                $products->orderBy('unit_price', 'asc');
                break;
            case 'price-desc':
                $products->orderBy('unit_price', 'desc');
                break;
            default:
                $products->orderBy('id', 'desc');
                break;
        }

        $products = filter_products($products)->with('taxes')->paginate(8);

        $productHtml = [];
        foreach ($products as $product) {
            $productHtml[] = '<div class="col custompad-10 has-transition z-1">' . view('frontend.partials.boxes.product_box_1', ['product' => $product])->render() . '</div>';
        }

        return response()->json([
            'products' => $productHtml,
            'next_page_url' => $products->nextPageUrl(),
            'has_more_pages' => $products->hasMorePages(),
        ]);
    }

    public function listing(Request $request)
    {
        return $this->index($request);
    }
    public function listingByCategory(Request $request, $category_slug)
    {
        $category = Category::where('slug', $category_slug)->first();
        if ($category != null) {
            return $this->index($request, $category->id);
        }
        abort(404);
    }
    public function listingByBrand(Request $request, $brand_slug)
    {
        $brand = Brand::where('slug', $brand_slug)->first();
        if ($brand != null) {
            return $this->index($request, null, $brand->id);
        }
        abort(404);
    }
    //Suggestional Search
    public function ajax_search(Request $request)
    {
        $keywords = array();
        $query = $request->search;
        $products = Product::where('published', 1)->where('tags', 'like', '%' . $query . '%')->get();
        foreach ($products as $key => $product) {
            foreach (explode(',', $product->tags) as $key => $tag) {
                if (stripos($tag, $query) !== false) {
                    if (sizeof($keywords) > 5) {
                        break;
                    } else {
                        if (!in_array(strtolower($tag), $keywords)) {
                            array_push($keywords, strtolower($tag));
                        }
                    }
                }
            }
        }
        $products_query = filter_products(Product::query());
        $products_query = $products_query->where('published', 1)
            ->where(function ($q) use ($query) {
                foreach (explode(' ', trim($query)) as $word) {
                    $q->where('name', 'like', '%' . $word . '%')
                        ->orWhere('tags', 'like', '%' . $word . '%')
                        ->orWhere('unit_price', 'like', '%' . $word . '%')
                        ->orWhereHas('product_translations', function ($q) use ($word) {
                            $q->where('name', 'like', '%' . $word . '%');
                        })
                        ->orWhereHas('stocks', function ($q) use ($word) {
                            $q->where('sku', 'like', '%' . $word . '%');
                        });
                }
            });
        $case1 = $query . '%';
        $case2 = '%' . $query . '%';
        $products_query->orderByRaw("CASE
                WHEN name LIKE '$case1' THEN 1
                WHEN name LIKE '$case2' THEN 2
                ELSE 3
                END");
        $products = $products_query->limit(5)->get();
        $categories = Category::where('name', 'like', '%' . $query . '%')->get()->take(8);
        $shops = Shop::whereIn('user_id', verified_sellers_id())->where('name', 'like', '%' . $query . '%')->get()->take(3);
        if (sizeof($keywords) > 0 || sizeof($categories) > 0 || sizeof($products) > 0 || sizeof($shops) > 0) {
            return view('frontend.partials.search_content', compact('products', 'categories', 'keywords', 'shops'));
        }
        return '0';
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $search = Search::where('query', $request->keyword)->first();
        if ($search != null) {
            $search->count = $search->count + 1;
            $search->save();
        } else {
            $search = new Search;
            $search->query = $request->keyword;
            $search->save();
        }
    }
}
