<?php
namespace App\Models;
use App\Models\Product;
use App\Models\ProductStock;
use App\Models\User;
use App\Traits\PreventDemoModeChanges;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Illuminate\Support\Str;
use Auth;
use Carbon\Carbon;
use Storage;
//class ProductsImport implements ToModel, WithHeadingRow, WithValidation
class ProductsImport implements ToCollection, WithHeadingRow, WithValidation, ToModel
{
    use PreventDemoModeChanges;
    private $rows = 0;
    public function collection(Collection $rows)
    {
        $canImport = true;
        $user = Auth::user();
        if ($user->user_type == 'seller' && addon_is_activated('seller_subscription')) {
            if (
                (count($rows) + $user->products()->count()) > $user->shop->product_upload_limit
                || $user->shop->package_invalid_at == null
                || Carbon::now()->diffInDays(Carbon::parse($user->shop->package_invalid_at), false) < 0
            ) {
                $canImport = false;
                flash(translate('Please upgrade your package.'))->warning();
            }
        }
        if ($canImport) {
            foreach ($rows as $row) {
                $approved = 1;
                if ($user->user_type == 'seller' && get_setting('product_approve_by_admin') == 1) {
                    $approved = 0;
                }

                $attribute_values = explode(', ', $row['attribute_values']);

                $productId = Product::create([
                    'name' => $row['name'],
                    'description' => $row['description'],
                    'added_by' => $user->user_type == 'seller' ? 'seller' : 'admin',
                    'user_id' => $user->user_type == 'seller' ? $user->id : User::where('user_type', 'admin')->first()->id,
                    'approved' => $approved,
                    'category_id' => $row['category_id'],
                    'tags' => $row['tags'],
                    'unit_price' => $row['unit_price'],
                    'meta_title' => $row['meta_title'],
                    'meta_description' => $row['meta_description'],
                    'digital' => $row['digital'],
                    'variations' => json_encode(array()),
                    'slug' => preg_replace('/[^A-Za-z0-9\-]/', '', str_replace(' ', '-', strtolower($row['slug']))) . '-' . Str::random(5),
                    'thumbnail_img' => $this->downloadThumbnail($row['thumbnail_img']),
                    'photos' => $this->downloadGalleryImages($row['photos']),
                    'attributes' => json_encode([$row['attribute_id'] . ""]),
                    'choice_options' => json_encode([['attribute_id' => $row['attribute_id'], 'values' => $attribute_values]]),
                    'event_city' => $row['event_location'], 
                    'event_start_date' => strtotime($row['event_start_date']),
                    'event_end_date' => strtotime($row['event_end_date']),
                ]);

                if (!empty($attribute_values)) {
                    foreach ($attribute_values as $value) {
                        ProductStock::create([
                            'product_id' => $productId->id,
                            'qty' => $row['current_stock'],
                            'price' => $row['unit_price'],
                            'sku' => '',
                            'variant' => $value,
                        ]);
                    }
                } else {
                    ProductStock::create([
                        'product_id' => $productId->id,
                        'qty' => $row['current_stock'],
                        'price' => $row['unit_price'],
                        'sku' => '',
                        'variant' => '',
                    ]);
                }

                if ($row['multi_categories'] != null) {
                    foreach (explode(',', $row['multi_categories']) as $category_id) {
                        ProductCategory::insert([
                            "product_id" => $productId->id,
                            "category_id" => $category_id
                        ]);
                    }
                }
            }
            flash(translate('Products imported successfully'))->success();
        }
    }
    public function model(array $row)
    {
        ++$this->rows;
    }
    public function getRowCount(): int
    {
        return $this->rows;
    }
    public function rules(): array
    {
        return [
            // Can also use callback validation rules
            'unit_price' => function ($attribute, $value, $onFailure) {
                if (!is_numeric($value)) {
                    $onFailure('Unit price is not numeric');
                }
            }
        ];
    }
    public function downloadThumbnail($url)
    {
        $base_url = url('/public/');

        try {
            $upload = new Upload;
            if (strpos($url, $base_url) === 0) {
                $url = substr($url, strlen($base_url));
                $upload->file_name = $url;
                $upload->user_id = Auth::user()->id;
            } else {
                $upload->external_link = $url;
            }
            $upload->type = 'image';
            $upload->save();
            return $upload->id;
        } catch (\Exception $e) {
        }
        return null;
    }
    public function downloadGalleryImages($urls)
    {
        $data = array();
        foreach (explode(',', str_replace(' ', '', $urls)) as $url) {
            $data[] = $this->downloadThumbnail($url);
        }
        return implode(',', $data);
    }
}
