<?php
namespace App\Services;
use Illuminate\Support\Facades\File;

class ImageCompressionService
{
    protected $imagePath;
    protected $uploadDir;
    protected $compressedDir;
    protected $quality;
    protected $maxWidth;

    public function __construct($imagePath = null)
    {
        $this->imagePath = public_path($imagePath);
        $this->uploadDir = public_path('uploads/all/');
        $this->compressedDir = public_path('uploads/all/');
        $this->quality = 80;
        $this->maxWidth = 1920;
        $this->createDirectories();
    }

    protected function createDirectories()
    {
        if (!File::isDirectory($this->uploadDir)) {
            File::makeDirectory($this->uploadDir, 0755, true);
        }
        if (!File::isDirectory($this->compressedDir)) {
            File::makeDirectory($this->compressedDir, 0755, true);
        }
    }

    public function compressImages()
    {
        if (!is_null($this->imagePath)) {
            return $this->compressImage($this->imagePath);
        } else {
            $images = File::glob($this->uploadDir . "*.{jpg,jpeg,png,webp}", GLOB_BRACE);
            foreach ($images as $originalPath) {
                $this->compressImage($originalPath);
            }
            return null;
        }
    }

    protected function compressImage($originalPath)
    {
        $info = getimagesize($originalPath);
        $mime = $info['mime'];
        switch ($mime) {
            case 'image/jpeg':
                $image = imagecreatefromjpeg($originalPath);
                break;
            case 'image/png':
                $image = imagecreatefrompng($originalPath);
                break;
            case 'image/webp':
                $image = imagecreatefromwebp($originalPath);
                break;
            default:
                throw new \Exception("Unsupported image type for conversion to WebP via GD.");
        }
        $image = $this->resizeImageIfNeeded($image);
        $pathInfo = pathinfo($originalPath);
        $compressedFileName = $pathInfo['filename'] . '.webp';
        $compressedPath = $this->compressedDir . $compressedFileName;
        if (!imagewebp($image, $compressedPath, $this->quality)) {
            throw new \Exception("Error converting to WebP format with GD.");
        }

        imagedestroy($image);

        return [
            'path' => 'uploads/all/' . $compressedFileName,
            'size' => filesize($compressedPath),
            'extension' => 'webp'
        ];
    }

    protected function resizeImageIfNeeded($image)
    {
        $width = imagesx($image);
        $height = imagesy($image);
        if ($width > $this->maxWidth) {
            $newHeight = (int) (($this->maxWidth / $width) * $height);
            $newImage = imagecreatetruecolor($this->maxWidth, $newHeight);
            imagecopyresampled($newImage, $image, 0, 0, 0, 0, $this->maxWidth, $newHeight, $width, $height);
            imagedestroy($image);
            return $newImage;
        }
        return $image;
    }

    public function deleteImage($imagePath)
    {
        $fullPath = public_path($imagePath);
        if (File::exists($fullPath)) {
            File::delete($fullPath);
            return "Image deleted successfully.";
        } else {
            throw new \Exception("Image not found at path: $fullPath");
        }
    }
}
