<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;

/**
 * Service class for sending notifications via OneSignal.
 */
class OneSignalService
{
    protected $client;
    protected $appId;
    protected $restApiKey;

    /**
     * Constructor to initialize OneSignal credentials.
     */
    public function __construct()
    {
        $this->client = new Client();
        $this->appId = config('services.onesignal.app_id');
        $this->restApiKey = config('services.onesignal.rest_api_key');
    }

    /**
     * Send a notification via OneSignal.
     *
     * @param string $title   Notification title.
     * @param string $message Notification message.
     * @param string $url     URL to open when notification is clicked.
     * @param string $image   Image URL for the notification.
     * @return array|bool     Response from OneSignal API or false on failure.
     */
    public function sendNotification($title, $message, $url = null, $image = null)
    {
        $fields = [
            'app_id' => $this->appId,
            'included_segments' => ['All'],
            'headings' => ['en' => $title],
            'contents' => ['en' => $message],
        ];

        if ($url) {
            $fields['url'] = $url;
        }

        if ($image) {
            $fields['chrome_web_image'] = $image;
        }

        try {
            $response = $this->client->post('https://onesignal.com/api/v1/notifications', [
                'headers' => [
                    'Authorization' => 'Basic ' . $this->restApiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => $fields,
            ]);

            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            Log::error('OneSignal API Error: ' . $e->getMessage());
            return false;
        }
    }
}
