<?php
namespace App\Services;
use CreaPackages\CombinationGenerate\Services\CombinationService;
use App\Models\AttributeValue;
use App\Models\Cart;
use App\Models\Category;
use App\Models\Color;
use App\Models\Product;
use App\Models\Review;
use App\Models\User;
use App\Models\Wishlist;
use App\Utility\ProductUtility;
use Combinations;
use Illuminate\Support\Str;
class ProductService
{
    // Helper functions
    private function get_approval_status()
    {
        $user_type = auth()->user()->user_type;
        if ('seller' === $user_type && 1 == get_setting('product_approve_by_admin')) {
            return 0;
        }
        return 1;
    }
    private function get_user_id()
    {
        return auth()->user()->user_type == 'seller' ? auth()->user()->id : User::where('user_type', 'admin')->first()->id;
    }
    private function process_general_information(&$product_data, $collection)
    {
        $product_data['type'] = isset($collection['type']) ? $collection['type'] : 'simple';
        $product_data['nature'] = isset($collection['nature']) ? $collection['nature'] : 'event';
        $product_data['name'] = isset($collection['name']) ? $collection['name'] : '';
        $product_data['description'] = isset($collection['description']) ? $collection['description'] : '';
        $product_data['flash_message'] = isset($collection['flash_message']) ? $collection['flash_message'] : '';
        $processed_tags = [];
        if (isset($collection['tags'])) {
            $tags = $collection['tags'];
            if (!empty($tags[0])) {
                foreach (json_decode($tags[0]) as $tag) {
                    $processed_tags[] = $tag->value;
                }
            }
        }
        $product_data['tags'] = implode(',', $processed_tags);
    }
    private function process_meta_information(&$product_data, $collection)
    {
        $product_data['meta_title'] = isset($collection['meta_title']) ? $collection['meta_title'] : $collection['name'];
        $product_data['meta_description'] = isset($collection['meta_description']) ? strip_tags($collection['meta_description']) : strip_tags($collection['description']);
        $product_data['meta_img'] = isset($collection['meta_img']) ? $collection['meta_img'] : $collection['thumbnail_img'];
    }
    private function process_product_category(&$product_data, $collection)
    {
        $product_data['category_id'] = isset($collection['category_id']) ? $collection['category_id'] : '';
    }

    private function process_product_brand(&$product_data, $collection)
    {
        $product_data['brand_id'] = isset($collection['brand_id']) ? $collection['brand_id'] : '';
    }

    private function process_images(&$product_data, $collection)
    {
        $product_data['thumbnail_img'] = isset($collection['thumbnail_img']) ? $collection['thumbnail_img'] : null;
        $product_data['photos'] = isset($collection['photos']) ? $collection['photos'] : null;
        $product_data['featured_img'] = isset($collection['featured_img']) ? $collection['featured_img'] : null;
        $product_data['banner_img'] = isset($collection['banner_img']) ? $collection['banner_img'] : null;
    }
    private function process_event_information(&$product_data, $collection)
    {
        $is_event = isset($collection['location']) && $collection['location'] != '' ? 1 : 0;
        $location = $collection['location'] ?? '';
        $address = $collection['address'] ?? '';
        if (isset($collection['event_date_range'])) {
            $date_range = explode(" to ", $collection['event_date_range']);
            $event_start_date = strtotime($date_range[0]);
            $event_end_date = strtotime($date_range[1]);
        } else {
            $event_start_date = $event_end_date = '';
        }
        $product_data['is_event'] = $is_event;
        $product_data['location'] = $location;
        $product_data['address'] = $address;
        $product_data['event_start_date'] = $event_start_date;
        $product_data['event_end_date'] = $event_end_date;

        $product_data['date_open_hour'] = $collection['date_open_hour'] ?? '';
        $product_data['date_open_minute'] = $collection['date_open_minute'] ?? '';

        $product_data['date_close_hour'] = $collection['date_close_hour'] ?? '';
        $product_data['date_close_minute'] = $collection['date_close_minute'] ?? '';
    }

    private function process_default_values(&$product_data, $collection)
    {
        if (isset($collection['default_sales'])) {
            $product_data['default_sales'] = $collection['default_sales'];
        }

        if (isset($collection['default_reviews_number'])) {
            $product_data['default_reviews_number'] = $collection['default_reviews_number'];
        }

        if (isset($collection['default_reviews_stars'])) {
            $product_data['default_reviews_stars'] = $collection['default_reviews_stars'];
        }
    }

    private function process_map_link(&$product_data, $collection)
    {
        if (isset($collection['map_link'])) {
            $product_data['map_link'] = $collection['map_link'];
        }

        if (isset($collection['map'])) {
            $product_data['map'] = $collection['map'];
        }
    }
    private function process_price(&$product_data, $collection)
    {
        $unit_price = $collection['unit_price'][0];
        if (isset($unit_price) && !is_null($unit_price)) {
            $product_data['unit_price'] = $unit_price;
        }
    }
    private function process_discount_dates(&$product_data, $collection)
    {
        if (isset($collection['date_range'][0]) && !is_null($collection['date_range'][0])) {
            $dates = explode(" to ", $collection['date_range'][0]);
            $product_data['discount_start_date'] = strtotime($dates[0]);
            $product_data['discount_end_date'] = strtotime($dates[1]);

            $unit_price = $collection['unit_price'][0];
            $product_data['discount'] = $collection['discount'];
            $product_data['discount_type'] = $collection['discount_type'];

            if ($product_data['discount_type'] == 'amount') {
                $discount_value = $product_data['discount'];
            } elseif ($product_data['discount_type'] == 'percent') {
                $discount_value = ($unit_price * $product_data['discount']) / 100;
            } else {
                $discount_value = 0;
            }

            $discounted_price = $unit_price - $discount_value;

            $product_data['discounted_price'] = $discounted_price;
        } else {
            $product_data['discount_start_date'] = null;
            $product_data['discount_end_date'] = null;
        }
    }
    private function process_attributes(&$product_data, $collection)
    {
        $options = ProductUtility::get_attribute_options($collection);
        $combinations = (new CombinationService())->generate_combination($options);
        if (count($combinations) > 0) {
            foreach ($combinations as $key => $combination) {
                $str = ProductUtility::get_combination_string($combination, $collection);
                unset($collection['price_' . str_replace('.', '_', $str)]);
                unset($collection['qty_' . str_replace('.', '_', $str)]);
            }
        }
        $choice_options = array();
        if (isset($collection['choice_no']) && $collection['choice_no']) {
            $str = '';
            $item = array();
            foreach ($collection['choice_no'] as $key => $no) {
                $str = 'choice_options_' . $no;
                $item['attribute_id'] = $no;
                $attribute_data = array();
                // foreach (json_decode($request[$str][0]) as $key => $eachValue) {
                foreach ($collection[$str] as $key => $eachValue) {
                    // array_push($data, $eachValue->value);
                    array_push($attribute_data, $eachValue);
                }
                unset($collection[$str]);
                $item['values'] = $attribute_data;
                array_push($choice_options, $item);
            }
        }
        $product_data['choice_options'] = json_encode($choice_options, JSON_UNESCAPED_UNICODE);
        if (isset($collection['choice_no']) && $collection['choice_no']) {
            $product_data['attributes'] = json_encode($collection['choice_no']);
            unset($collection['choice_no']);
        } else {
            $product_data['attributes'] = json_encode(array());
        }
    }

    private function create_default_reviews($product_id, $reviews_total, $reviews_stars)
    {
        Review::where('product_id', $product_id)->delete();

        $reviewers = User::where('user_type', 'reviewer')->inRandomOrder()->get();

        foreach ($reviewers->take($reviews_total) as $reviewer) {
            $review = new Review;
            $review->product_id = $product_id;
            $review->user_id = $reviewer->id;
            $review->rating = $reviews_stars;
            $review->comment = '';
            $review->viewed = '0';
            $review->save();
        }
    }

    public function store(array $data)
    {
        $product_data = [];
        $collection = collect($data);

        $product_data['approved'] = $this->get_approval_status();
        $product_data['user_id'] = $this->get_user_id();
        $product_data['added_by'] = isset($collection['added_by']) ? $collection['added_by'] : '';
        $product_data['digital'] = isset($collection['digital']) ? $collection['digital'] : 0;
        // Process general information
        $this->process_general_information($product_data, $collection);
        $slug = Str::slug($product_data['name']);
        $same_slug_count = Product::where('slug', 'LIKE', $slug . '%')->count();
        $product_data['slug'] = $same_slug_count ? "{$slug}-" . ($same_slug_count + 1) : $slug;
        // Process meta information
        $this->process_meta_information($product_data, $collection);
        // Process products category
        $this->process_product_category($product_data, $collection);
        // Process product brand
        $this->process_product_brand($product_data, $collection);
        // Process images 
        $this->process_images($product_data, $collection);
        // Process event information
        $this->process_event_information($product_data, $collection);
        // Process default values
        $this->process_default_values($product_data, $collection);
        // Process map link
        $this->process_map_link($product_data, $collection);
        // Process price
        $this->process_price($product_data, $collection);
        // Process discount dates
        $this->process_discount_dates($product_data, $collection);
        // Process attributes
        $this->process_attributes($product_data, $collection);
        $_product_data = collect($product_data)->toArray();
        $product = Product::create($_product_data);

        if (!is_null($product->default_reviews_number) && !is_null($product->default_reviews_stars)) {
            $this->create_default_reviews($product->id, $product->default_reviews_number, $product->default_reviews_stars);
        }

        return $product;
    }
    public function update(array $data, Product $product)
    {
        $product_data = [];
        $collection = collect($data);

        // $product_data['added_by'] = isset($collection['added_by']) ? $collection['added_by'] : '';
        // $product_data['digital'] = isset($collection['digital']) ? $collection['digital'] : 0;
        $product_data['approved'] = $this->get_approval_status();
        // Process general information
        $this->process_general_information($product_data, $collection);
        $slug = Str::slug($collection['slug']);
        $same_slug_count = Product::where('slug', 'LIKE', $slug . '%')->count();
        $product_data['slug'] = $same_slug_count > 1 ? "{$slug}-" . ($same_slug_count + 1) : $slug;

        $update_reviews = false;
        if (is_null($product->default_reviews_number) && is_null($product->default_reviews_stars)) {
            $update_reviews = true;
        }

        // Process meta information
        $this->process_meta_information($product_data, $collection);
        // Process products category
        $this->process_product_category($product_data, $collection);
        // Process product brand
        $this->process_product_brand($product_data, $collection);
        // Process images 
        $this->process_images($product_data, $collection);
        // Process event information
        $this->process_event_information($product_data, $collection);
        // Process default valudes
        $this->process_default_values($product_data, $collection);
        // Process map link
        $this->process_map_link($product_data, $collection);
        // Process price
        $this->process_price($product_data, $collection);
        // Process discount dates
        $this->process_discount_dates($product_data, $collection);
        // Process attributes
        $this->process_attributes($product_data, $collection);
        $_product_data = collect($product_data)->toArray();
        $product->update($_product_data);

        // if ($update_reviews) {
        //     if (!is_null($product->default_reviews_number) && !is_null($product->default_reviews_stars)) {
                $this->create_default_reviews($product->id, $product->default_reviews_number, $product->default_reviews_stars);
        //     }
        // }

        return $product;
    }
    public function product_duplicate_store($product)
    {
        $product_new = $product->replicate();
        $product_new->slug = $product_new->slug . '-' . Str::random(5);
        $product_new->approved = (get_setting('product_approve_by_admin') == 1 && $product->added_by != 'admin') ? 0 : 1;
        $product_new->save();
        return $product_new;
    }
    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->product_translations()->delete();
        $product->categories()->detach();
        $product->stocks()->delete();
        $product->taxes()->delete();
        $product->wishlists()->delete();
        $product->carts()->delete();
        $product->frequently_bought_products()->delete();
        $product->last_viewed_products()->delete();
        $product->flash_deal_products()->delete();
        Product::destroy($id);
    }
    public function product_search(array $data)
    {
        $collection = collect($data);
        $auth_user = auth()->user();
        $productType = $collection['product_type'];
        $products = Product::query();
        if ($collection['category'] != null) {
            $category = Category::with('childrenCategories')->find($collection['category']);
            $products = $category->products();
        }
        $products = in_array($auth_user->user_type, ['admin', 'staff']) ? $products->where('products.added_by', 'admin') : $products->where('products.user_id', $auth_user->id);
        $products->where('published', '1')->where('auction_product', 0)->where('approved', '1');
        if ($productType == 'physical') {
            $products->where('digital', 0)->where('wholesale_product', 0);
        } elseif ($productType == 'digital') {
            $products->where('digital', 1);
        } elseif ($productType == 'wholesale') {
            $products->where('wholesale_product', 1);
        }
        if ($collection['product_id'] != null) {
            $products->where('id', '!=', $collection['product_id']);
        }
        if ($collection['search_key'] != null) {
            $products->where('name', 'like', '%' . $collection['search_key'] . '%');
        }
        return $products->limit(10)->get();
    }
    public function setCategoryWiseDiscount(array $data)
    {
        $auth_user = auth()->user();
        $discount_start_date = null;
        $discount_end_date = null;
        if (isset($data['date_range'])) {
            $date_var = explode(" to ", $data['date_range']);
            $discount_start_date = strtotime($date_var[0]);
            $discount_end_date = strtotime($date_var[1]);
        }
        $seller_product_discount = isset($data['seller_product_discount']) ? $data['seller_product_discount'] : null;
        $admin_id = User::where('user_type', 'admin')->first()->id;
        $products = Product::where('category_id', $data['category_id'])->where('auction_product', 0);
        if (in_array($auth_user->user_type, ['admin', 'staff']) && $seller_product_discount == 0) {
            $products = $products->where('user_id', $admin_id);
        } elseif ($auth_user->user_type == 'seller') {
            $products = $products->where('user_id', $auth_user->id);
        }
        $products->update([
            'discount' => $data['discount'],
            'discount_type' => 'percent',
            'discount_start_date' => $discount_start_date,
            'discount_end_date' => $discount_end_date,
        ]);
        return 1;
    }
}